"""Hyperparameter tuning infrastructure.

This is the hyperparameter tuning component of `cmlkit`.
At the moment, it contains a wrapper for the `hyperopt` package,
which works with a custom parallelised, asynchronous optimisation
infrastructure. Important features are:

- Robust parallelisation (on a single computer) with timeouts (i.e. we can
  deal with unstable external components that don't terminate).
- Resuming/replay: Runs are saved as a series of steps, so they can always
  be recovered, replayed and continued. The steps are saved as the optimisation
  proceeds, and writes are atomic, so very little can go wrong.
- Caching of evaluations: no redundant evaluations are undertaken.
- Flexible infrastructure: Computation of losses, the types of models evaluated,
  and the stopping criteria can be customised.
- Async architecture: Recognising that model evaluation cannot be expected to take
  the same time, so everything happens asynchronously.
- Transparent, configurable error treatment: Errors can be optionally caught and
  reported.

***

Conceptually, we separate the task of optimising into multiple components:

1) A Search, which makes suggestions (based on previous results),
2) An Evaluator which computes losses, and
3) A Run instance, that takes care of the execution, saving results, etc.

Users are mainly expected to implement their own Evaluators, and to
provide a suitably-formatted dictionary describing the search space
to the Search instance.

The basic nomenclature is as follows:
- Suggestion: Emitted by the search, it represents one possible model that
  should be tried. Normally a config, but in general any (serialisable) dict.
- Trial: The task of performing the evaluation of such a suggestion.
- Evaluation: The unique task of evaluating a particular dictionary/config.
  Since Searches are free to make the same suggestion multiple time, this is
  not identical to the concept of a Trial. Think of it as the "de-duplicated
  version" of a Trial.
- tid/eid: Trial ID and Evaluation ID. Trial IDs are generated by the Search and
  must only occur once during a search, Evaluation IDs are simple the hash of a
  suggestion.
- tape: The "trajectory" of the optimisation, a series of steps that, if retraced
  in the same order with the same arguments, yield an identical optimisation state.

Note that we prefer to say "search" as opposed to "optimisation" to reflect
the fact that this is fundamentally different from "normal" optimisation methods:
There are no guarantees/expectations on the order in which candidate models are
evaluated, everything is asynchronous in nature, steps are not particularly sequential.

For implementation details, please read the docstrings of the various
components.

There will also be a readme with some examples.

***

## Interfaces

If you implement custom Evaluators/Searches, here is the interface
that they need to implement.

Search:
    suggest() -> tid, suggestion.
        tid must be unique. it identifies this suggestion.
        suggestion must be a dict and should be a valid config.
    submit(tid, error=False, loss=None, var=None).
        tid must match a previously suggested one.
        if the trial failed, error must be true, loss and var are ignored.
        if the trial succeeded, error must be false, loss is required and var is optional.

    Searches must be deterministic with suggest and submit are performed in the same order
    with the same arguments.

Evaluator:
    __call__(config) -> result.
        result must contain the keys "loss" and "duration".
        it can contain the keys "var" and "refined_config".
        additional keys are ignored.
        exceptions must be raised, not caught.

"""

from .search import Hyperopt
from .run import Run
from .evaluators import TuneEvaluatorHoldout

components = [Hyperopt, Run, TuneEvaluatorHoldout]
